﻿using System;
using System.IO;
using System.Net;
using System.Xml;
using System.Diagnostics;
using System.Security.Cryptography.X509Certificates;
using Service_Manager.CIS.Requests;
using System.Windows.Forms;

namespace Service_Manager.CIS
{
    /// <summary>
    /// Koristi se za komunikaciju sa CIS-om.</summary>
    /// <remarks>
    /// Ovo je osnovna klasa čija je namjena pozivanje web metoda CIS-a: Echo, RacunZahtjev i PoslovniProstorZahtjev.
    /// </remarks>
    public class CentralniInformacijskiSustav
    {
        #region Events
        /// <summary>
        /// "Okida" se neposredno prije poziva web servisa. Cancel postavljen na true će prekinuti poziv.</summary>
        public event EventHandler<CentralniInformacijskiSustavEventArgs> SoapMessageSending;
        /// <summary>
        /// "Okida" se neposredno nakon poziva web servisa.</summary>
        public event EventHandler<EventArgs> SoapMessageSent;
        #endregion

        #region Fields
        /// <summary>
        /// Sadrži URL do web servisa, u ovom trenutku se radi o službenom testnom okruženju.</summary>
        /// <remarks>
        /// U finalnoj verziji ova će se adresa svakako mijenjati.
        /// </remarks>
        private const string cisUrl = "https://cistest.apis-it.hr:8449/FiskalizacijaServiceTest";

        /// <summary>
        /// Sadrži default timeout = 10000
        /// </summary>
        private const int timeout = 10000;
        #endregion



        #region Poslovni prostor
        /// <summary>
        /// Koristi se za slanje informacija o poslovnom prostoru (PoslovniProstorZahtjev).</summary>
        /// <param name="poslovniProstor">Objekt tipa Schema.PoslovniProstorType koji sadrži informacije o poslovnom prostoru.</param>
        /// <param name="certificateSubject">Naziv certifikata koji se koristi, na primjer "FISKAL 1".</param>
        /// <example>
        ///  Raverus.FiskalizacijaDEV.Schema.PoslovniProstorType poslovniProstor = new Schema.PoslovniProstorType();
        ///  XmlDocument doc = cis.PosaljiPoslovniProstor(poslovniProstor, "FISKAL 1");
        /// </example>
        /// <returns>
        /// Vraća XmlDocument koji sadrži XML poruku vraćeno od CIS-a. U slučaju greške, vraća null.</returns>
        public XmlDocument PosaljiPoslovniProstor(Schema.PoslovniProstorType poslovniProstor, string certificateSubject)
        {
            XmlDocument racunOdgovor = null;
            
            Schema.PoslovniProstorZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajPoslovniProstorZahtjev(poslovniProstor);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajPoslovniProstorZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }

        public XmlDocument PosaljiPoslovniProstor(Schema.PoslovniProstorType poslovniProstor, X509Certificate2 certifikat)
        {
            XmlDocument racunOdgovor = null;

            Schema.PoslovniProstorZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajPoslovniProstorZahtjev(poslovniProstor);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajPoslovniProstorZahtjev(zahtjev);

            Popratne_funkcije.Potpisivanje.PotpisiXmlDokument(zahtjevXml, certifikat);
            Popratne_funkcije.XmlDokumenti.DodajSoapEnvelope(ref zahtjevXml);

            racunOdgovor = SendSoapMessage(zahtjevXml);


            return racunOdgovor;
        }

        public XmlDocument PosaljiPoslovniProstor(Schema.PoslovniProstorType poslovniProstor, string certificateSubject, StoreLocation storeLocation, StoreName storeName)
        {
            // prema sugestiji dkustec: http://fiskalizacija.codeplex.com/workitem/693
            XmlDocument racunOdgovor = null;

            Schema.PoslovniProstorZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajPoslovniProstorZahtjev(poslovniProstor);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajPoslovniProstorZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, storeLocation, storeName, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }

        public XmlDocument PosaljiPoslovniProstor(Schema.PoslovniProstorType poslovniProstor, string certificateSubject, DateTime datumVrijeme)
        {
            XmlDocument racunOdgovor = null;

            Schema.PoslovniProstorZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajPoslovniProstorZahtjev(poslovniProstor, datumVrijeme);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajPoslovniProstorZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }

        public XmlDocument PosaljiPoslovniProstor(Schema.PoslovniProstorType poslovniProstor, X509Certificate2 certifikat, DateTime datumVrijeme)
        {
            XmlDocument racunOdgovor = null;

            Schema.PoslovniProstorZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajPoslovniProstorZahtjev(poslovniProstor, datumVrijeme);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajPoslovniProstorZahtjev(zahtjev);

            Popratne_funkcije.Potpisivanje.PotpisiXmlDokument(zahtjevXml, certifikat);
            Popratne_funkcije.XmlDokumenti.DodajSoapEnvelope(ref zahtjevXml);

            racunOdgovor = SendSoapMessage(zahtjevXml);


            return racunOdgovor;
        }

        public XmlDocument PosaljiPoslovniProstor(Schema.PoslovniProstorType poslovniProstor, string certificateSubject, StoreLocation storeLocation, StoreName storeName, DateTime datumVrijeme)
        {
            // prema sugestiji dkustec: http://fiskalizacija.codeplex.com/workitem/693
            XmlDocument racunOdgovor = null;

            Schema.PoslovniProstorZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajPoslovniProstorZahtjev(poslovniProstor, datumVrijeme);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajPoslovniProstorZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, storeLocation, storeName, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }
        #endregion


        #region Račun
        /// <summary>
        /// Koristi se za slanje informacija o računu (RacunZahtjev).</summary>
        /// <param name="racun">Objekt tipa Schema.RacunType koji sadrži informacije o računu.</param>
        /// <param name="certificateSubject">Naziv certifikata koji se koristi, na primjer "FISKAL 1".</param>
        /// <example>
        ///  Raverus.FiskalizacijaDEV.Schema.RacunType racun = new Schema.RacunType();
        ///  XmlDocument doc = cis.PosaljiRacun(racun, "FISKAL 1");
        /// </example>
        /// <returns>
        /// Vraća XmlDocument koji sadrži XML poruku vraćeno od CIS-a. U slučaju greške, vraća null.</returns>
        public XmlDocument PosaljiRacun(Schema.RacunType racun, string certificateSubject)
        {
            XmlDocument racunOdgovor = null;

            Schema.RacunZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajRacunZahtjev(racun);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajRacunZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }

        public XmlDocument PosaljiRacun(Schema.RacunType racun, X509Certificate2 certifikat)
        {
            XmlDocument racunOdgovor = null;


            Schema.RacunZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajRacunZahtjev(racun);


            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajRacunZahtjev(zahtjev);


            Popratne_funkcije.Potpisivanje.PotpisiXmlDokument(zahtjevXml, certifikat);


            Popratne_funkcije.XmlDokumenti.DodajSoapEnvelope(ref zahtjevXml);

            MessageBox.Show("Šaljem");

            racunOdgovor = SendSoapMessage(zahtjevXml);


            return racunOdgovor;
        }

        public XmlDocument PosaljiRacun(Schema.RacunType racun, string certificateSubject, StoreLocation storeLocation, StoreName storeName)
        {
            XmlDocument racunOdgovor = null;

            Schema.RacunZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajRacunZahtjev(racun);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajRacunZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, storeLocation, storeName, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }

        public XmlDocument PosaljiRacun(Schema.RacunType racun, string certificateSubject, DateTime datumVrijeme)
        {
            XmlDocument racunOdgovor = null;

            Schema.RacunZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajRacunZahtjev(racun, datumVrijeme);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajRacunZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }

        public XmlDocument PosaljiRacun(Schema.RacunType racun, X509Certificate2 certifikat, DateTime datumVrijeme)
        {
            XmlDocument racunOdgovor = null;

            Schema.RacunZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajRacunZahtjev(racun, datumVrijeme);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajRacunZahtjev(zahtjev);

            Popratne_funkcije.Potpisivanje.PotpisiXmlDokument(zahtjevXml, certifikat);
            Popratne_funkcije.XmlDokumenti.DodajSoapEnvelope(ref zahtjevXml);

            racunOdgovor = SendSoapMessage(zahtjevXml);


            return racunOdgovor;
        }

        public XmlDocument PosaljiRacun(Schema.RacunType racun, string certificateSubject, StoreLocation storeLocation, StoreName storeName, DateTime datumVrijeme)
        {
            XmlDocument racunOdgovor = null;

            Schema.RacunZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajRacunZahtjev(racun, datumVrijeme);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajRacunZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, storeLocation, storeName, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }
        #endregion


        #region ProvjeraZahtjev
        /// <summary>
        /// Koristi se za slanje informacija o provjeri računu (ProvjeraZahtjev).</summary>
        /// <param name="racun">Objekt tipa Schema.RacunType koji sadrži informacije o računu.</param>
        /// <param name="certificateSubject">Naziv certifikata koji se koristi, na primjer "FISKAL 1".</param>
        /// <returns>
        /// Vraća XmlDocument koji sadrži XML poruku vraćeno od CIS-a. U slučaju greške, vraća null.</returns>
        public XmlDocument PosaljiProvjeruRacuna(Schema.RacunType racun, string certificateSubject)
        {
            XmlDocument provjeraOdgovor = null;

            Schema.ProvjeraZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajProvjeraZahtjev(racun);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajProvjeraZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, ref provjeraOdgovor, zahtjevXml);

            return provjeraOdgovor;
        }

        public XmlDocument PosaljiProvjeruRacuna(Schema.RacunType racun, X509Certificate2 certifikat)
        {
            XmlDocument racunOdgovor = null;

            Schema.ProvjeraZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajProvjeraZahtjev(racun);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajProvjeraZahtjev(zahtjev);

            Popratne_funkcije.Potpisivanje.PotpisiXmlDokument(zahtjevXml, certifikat);
            Popratne_funkcije.XmlDokumenti.DodajSoapEnvelope(ref zahtjevXml);

            racunOdgovor = SendSoapMessage(zahtjevXml);


            return racunOdgovor;
        }

        public XmlDocument PosaljiProvjeruRacuna(Schema.RacunType racun, string certificateSubject, StoreLocation storeLocation, StoreName storeName)
        {
            XmlDocument racunOdgovor = null;

            Schema.ProvjeraZahtjev zahtjev = Popratne_funkcije.XmlDokumenti.KreirajProvjeraZahtjev(racun);
            XmlDocument zahtjevXml = Popratne_funkcije.XmlDokumenti.SerijalizirajProvjeraZahtjev(zahtjev);

            PosaljiZahtjev(certificateSubject, storeLocation, storeName, ref racunOdgovor, zahtjevXml);

            return racunOdgovor;
        }

        public XmlDocument PosaljiProvjeruRacuna(XmlDocument zahtjevXml)
        {
            XmlDocument provjeraOdgovor = SendSoapMessage(zahtjevXml);

            return provjeraOdgovor;
        }


        #endregion



        #region Echo
        /// <summary>
        /// Koristi se za slanje ECHO poruke u CIS.</summary>
        /// <param name="poruka">Tekst poruke koja se šalje, na primjer 'test' ili 'test poruka' ili sl. Ukoliko se radi o praznom stringu (""), tada će tekst poruke biti 'echo test'.</param>
        /// <example>
        ///  Raverus.FiskalizacijaDEV.CentralniInformacijskiSustav cis = new CentralniInformacijskiSustav();
        ///  XmlDocument doc = cis.PosaljiEcho("");
        /// </example>
        /// <returns>
        /// Vraća XmlDocument koji sadrži XML poruku vraćeno od CIS-a. U slučaju greške, vraća null.</returns>
        public XmlDocument PosaljiEcho(string poruka)
        {
            XmlDocument echoOdgovor = null;

            XmlDocument echoZahtjev = Popratne_funkcije.XmlDokumenti.DohvatiPorukuEchoZahtjev(poruka);
            if (echoZahtjev != null)
            {
                echoOdgovor = new XmlDocument();
                echoOdgovor = SendSoapMessage(echoZahtjev);
            }


            return echoOdgovor;
        }

        /// <summary>
        /// Koristi se za slanje ECHO poruke u CIS.</summary>
        /// <remarks>
        /// Namjena je ove metode da jednostavnim pozivom utvrdite da li servis radi ili ne.
        /// </remarks>
        /// <example>
        ///  
        ///  
        /// </example>
        /// <returns>
        /// Vraća True ukoliko je sve u redu i ukoliko je CIS vratio isti tekst poruke koji je i poslan. U suprotnom vraća False.</returns>
        public bool Echo()
        {
            return Echo("");
        }

        /// <summary>
        /// Koristi se za slanje ECHO poruke u CIS.</summary>
        /// <remarks>
        /// Namjena je ove metode da jednostavnim pozivom utvrdite da li servis radi ili ne.
        /// </remarks>
        /// <param name="poruka">Tekst poruke koja se šalje, na primjer 'test' ili 'test poruka' ili sl. Ukoliko se radi o praznom stringu (""), tada će tekst poruke biti 'echo test'.</param>
        /// <example>
        ///  
        ///  
        /// </example>
        /// <returns>
        /// Vraća True ukoliko je sve u redu i ukoliko je CIS vratio isti tekst poruke koji je i poslan. U suprotnom vraća False.</returns>
        public bool Echo(string poruka)
        {
            bool echo = false;

            XmlDocument echoOdgovor = PosaljiEcho(poruka);
            if (echoOdgovor != null && echoOdgovor.DocumentElement != null)
            {
                string odgovor = echoOdgovor.DocumentElement.InnerText.Trim();

                Popratne_funkcije.Razno.FormatirajEchoPoruku(ref poruka);

                if (poruka == odgovor)
                    echo = true;
            }

            return echo;
        }
        #endregion


        #region Razno
        /// <summary>
        /// Koristi se za slanje SOAP poruke u CIS.</summary>
        /// <remarks>
        /// XML dokument koji se šalje mora biti u skladu sa Tehničkom dokumentacijom.
        /// Namjena je ove metode da sami pripremite XML poruku, stavite SOAP zaglavlje, potpišete je i zatim je pošaljete koristeći ovu metodu.
        /// </remarks>
        /// <param name="soapPoruka">XML dokument koji šaljete u CIS.</param>
        /// <example>
        ///  
        ///  
        /// </example>
        /// <returns>
        /// Vraća XmlDocument koji sadrži XML poruku vraćeno od CIS-a. U slučaju greške, vraća null.</returns>
        public XmlDocument PosaljiSoapPoruku(XmlDocument soapPoruka)
        {
            return SendSoapMessage(soapPoruka);
        }

        public XmlDocument PosaljiSoapPoruku(string soapPoruka)
        {
            return SendSoapMessage(Popratne_funkcije.XmlDokumenti.UcitajXml(soapPoruka));
        }
        #endregion


        #region Private
        private XmlDocument SendSoapMessage(XmlDocument soapMessage)
        {
            XmlDocument responseSoapMessage = null;
            OdgovorGreska = null;


            if (SoapMessageSending != null)
            {
                CentralniInformacijskiSustavEventArgs ea = new CentralniInformacijskiSustavEventArgs() { SoapMessage = soapMessage };
                SoapMessageSending(this, ea);

                if (ea.Cancel)
                    return responseSoapMessage;
            }

            SnimanjeDatoteka(NazivMapeZahtjev, soapMessage);


            try
            {
                if (string.IsNullOrEmpty(CisUrl))
                    CisUrl = cisUrl;

                Requests.Request req = null;

                if (TimeOut == 0)
                    TimeOut = timeout;

                MessageBox.Show("Tražim pem path");
                if (PemPath != null)
                    req = new Requests.CurlWebRequest(PemPath);
                else
                    throw new ArgumentNullException("PemPath", "Na Win XP putanja do pem datoteke ne smije biti null vrijednost!");


                string response = string.Empty;

                MessageBox.Show("Request.send");

                if (!req.send(soapMessage.InnerXml, CisUrl, TimeOut, out response))
                    throw new Exception("Račun nije fiskaliziran.");

                responseSoapMessage = new XmlDocument();
                responseSoapMessage.PreserveWhitespace = true;
                responseSoapMessage.LoadXml(response);

            

                SnimanjeDatoteka(NazivMapeOdgovor, responseSoapMessage);

                if (SoapMessageSent != null)
                {
                    EventArgs ea = new EventArgs();
                    SoapMessageSent(this, ea);
                }
            }
            catch (CurlException ex)
            {
                
                if (!string.IsNullOrEmpty(ex.Response))
                {

                    OdgovorGreska = new XmlDocument();
                    OdgovorGreska.LoadXml(ex.Response);

                    Trace.TraceError("Greška kod slanja SOAP poruke. Primljen odgovor od CIS-a, detalji greške u CentralniInformacijskiSustav. OdgovorGreska.");

                    throw new Exception(ex.Message + " " + ex.Response, ex);

                }
                else
                {
                    Trace.TraceError(String.Format("Greška kod slanja SOAP poruke. Poruka greške: {1}", ex.Message));
                    throw new Exception(ex.Message);
                }          

            }
            

            catch (System.Web.Services.Protocols.SoapException ex)
            {
                OdgovorGreskaStatus = null;

                if (ex.Detail != null)
                {
                    OdgovorGreska = new XmlDocument();
                    OdgovorGreska.LoadXml(string.Format(@"<?xml version=""1.0"" encoding=""UTF - 8""?><soap:Envelope xmlns:soap=""http://schemas.xmlsoap.org/soap/envelope/"" xmlns:xsd=""http://www.w3.org/2001/XMLSchema"">{0}</soap:Envelope>", ex.Detail.InnerXml));

                    Trace.TraceError("Greška kod slanja SOAP poruke. Primljen odgovor od CIS-a, detalji greške u CentralniInformacijskiSustav.OdgovorGreska.");
                    throw ex;
                }
                else
                {
                    Trace.TraceError(String.Format("Greška kod slanja SOAP poruke. Status greške (prema http://msdn.microsoft.com/en-us/library/system.net.webexceptionstatus.aspx): {0}. Poruka greške: {1}", ex.Message, ex.Message));
                    throw ex;
                }
            }

            catch (Exception ex)
            {
                Trace.TraceError(String.Format("Greška kod slanja SOAP poruke: {0}", ex.Message));
                throw ex;
            }


            return responseSoapMessage;
        }

        private void SnimanjeDatoteka(string mapa, XmlDocument dokument)
        {
            if (!string.IsNullOrEmpty(mapa) && dokument != null)
            {
                Popratne_funkcije.TipDokumentaEnum tipDokumenta = Popratne_funkcije.XmlDokumenti.OdrediTipDokumenta(dokument);

                if (tipDokumenta != Popratne_funkcije.TipDokumentaEnum.Nepoznato)
                {
                    DirectoryInfo di = Popratne_funkcije.Razno.GenerirajProvjeriMapu(mapa);
                    if (di != null)
                    {
                        string file = "";

                        if (!NazivAutoGeneriranje)
                            file = Path.Combine(mapa, String.Format("{0}.xml", tipDokumenta));
                        else
                        {
                            string uuid = Popratne_funkcije.XmlDokumenti.DohvatiUuid(dokument, tipDokumenta);
                            if (!string.IsNullOrEmpty(uuid))
                                file = Path.Combine(mapa, String.Format("{0}_{1}.xml", tipDokumenta, uuid));
                        }

                        if (!string.IsNullOrEmpty(file))
                            Popratne_funkcije.XmlDokumenti.SnimiXmlDokumentDatoteka(dokument, file);
                    }
                }
            }
        }

        private void PosaljiZahtjev(string certificateSubject, ref XmlDocument racunOdgovor, XmlDocument zahtjevXml)
        {
            if (zahtjevXml != null && !string.IsNullOrEmpty(zahtjevXml.InnerXml))
            {
                X509Certificate2 certificate = Popratne_funkcije.Potpisivanje.DohvatiCertifikat(certificateSubject);
                if (certificate != null)
                {
                    Popratne_funkcije.Potpisivanje.PotpisiXmlDokument(zahtjevXml, certificate);
                    Popratne_funkcije.XmlDokumenti.DodajSoapEnvelope(ref zahtjevXml);

                    racunOdgovor = SendSoapMessage(zahtjevXml);
                }
            }
        }

        private void PosaljiZahtjev(string certificateSubject, StoreLocation storeLocation, StoreName storeName, ref XmlDocument racunOdgovor, XmlDocument zahtjevXml)
        {
            // prema sugestiji dkustec: http://fiskalizacija.codeplex.com/workitem/693
            if (zahtjevXml != null && !string.IsNullOrEmpty(zahtjevXml.InnerXml))
            {
                X509Certificate2 certificate = Popratne_funkcije.Potpisivanje.DohvatiCertifikat(certificateSubject, storeLocation, storeName);
                if (certificate != null)
                {
                    Popratne_funkcije.Potpisivanje.PotpisiXmlDokument(zahtjevXml, certificate);
                    Popratne_funkcije.XmlDokumenti.DodajSoapEnvelope(ref zahtjevXml);

                    racunOdgovor = SendSoapMessage(zahtjevXml);
                }
            }
        }
        #endregion




        #region Properties
        /// <summary>
        /// Naziv mape (foldera) u koji će se spremati XML dokumenti za zahtjeve. Ukoliko vrijednost nije postavljena, dokumenti se neće snimati.
        /// </summary>
        public string NazivMapeZahtjev { get; set; }

        /// <summary>
        /// Naziv mape (foldera) u koji će se spremati XML dokumenti za odgovore. Ukoliko vrijednost nije postavljena, dokumenti se neće snimati.
        /// </summary>
        public string NazivMapeOdgovor { get; set; }

        /// <summary>
        /// Određuje da li se naziv generira automatski koristeći UUID ili datoteka uvijek ima isti naziv
        /// </summary>
        /// <remarks>
        /// Ako je vrijednost true, naziv datoteke će biti određen koristeći naziv tipa dokumenta i UUID-a, ako je false naziv datoteke će biti uvijek isti i biti će određen tipom dokumenta.
        /// Ne koristi se ukoliko NazivMapeZahtjev odnosno NazivMapeOdgovor nisu postavljeni na odgovarajuću vrijednost.
        /// Nema smisla postavljati na TRUE za ECHO.
        /// </remarks>
        public bool NazivAutoGeneriranje { get; set; }

        /// <summary>
        /// Ukoliko je CIS vratio odgovor i ukoliko taj odgovor sadrži tehničkom specifikacijom propisanu grešku, tada OdgovorGreska sadži vraćenu XML poruku. U suprotnom, vrijednost je NULL.
        /// </summary>
        public XmlDocument OdgovorGreska { get; set; }

        /// <summary>
        /// Vraća WebExceptionStatus greške (http://msdn.microsoft.com/en-us/library/system.net.webexceptionstatus.aspx). U suprotnom, vrijednost je NULL.
        /// </summary>
        public WebExceptionStatus? OdgovorGreskaStatus { get; set; }

        /// <summary>
        /// Vrijednost, u milisekundama, za HttpWebRequest.TimeOut, odnosno, za TimeOut kod komunikacije sa CIS web servisom.
        /// Ako je vrijednost 0 (nula), property se ignorira (ne postavlja se vrijednost za HttpWebRequest.TimeOut).
        /// http://msdn.microsoft.com/query/dev10.query?appId=Dev10IDEF1&l=EN-US&k=k(SYSTEM.NET.HTTPWEBREQUEST.TIMEOUT)&rd=true
        /// </summary>
        public int TimeOut { get; set; }

        /// <summary>
        /// Adresa CIS web servisa; ako vrijednost nije postavljena, koristi se trenutna adresa TEST CIS web servisa koja je u službenoj upotrebi.
        /// </summary>
        public string CisUrl { get; set; }


        public string PemPath { get; set; }
        #endregion
    }
}


